' Wordle for CMM2
' Rev 1.0.0 William M Leue 9-Jan-2022

option default integer
option base 1

' Constants
const WLEN = 5
const TLEN = WLEN+2
const NUMROWS = 6
const NDWRDS = 5757
const NALPHA = 26
const dict$ = "5letterwords.txt"

' Layout params
const FWIDTH = 24
const FHEIGHT = 32
const FBGAP = 3
const GAP =   3
const BWIDTH = FBGAP+FWIDTH+FBGAP
const BHEIGHT = FBGAP+FHEIGHT+FBGAP
const PSWIDTH = WLEN*FWIDTH + (WLEN-1)*GAP
const PSHEIGHT = NUMROWS*FHEIGHT + (NUMROWS-1)*GAP
const AWIDTH = (NALPHA\2)*FWIDTH + (NALPHA\2-1)*GAP
const AHEIGHT = 2*FHEIGHT + GAP
const CTRX = MM.HRES\2
const PSX = CTRX - PSWIDTH\2
const PSY = 100
const ALX = CTRX - AWIDTH\2
const ALY = 400
const FNUM = 5

' Keystroke values
const ENTER  = 13
const ESC    = 27
const LCA    = 97
const LCZ    = 122
const CAPA   = 65
const CAPZ   = 90
const BACK   = 8
const HOME   = 134

' Globals
theWord$ = ""
dim rowContent(WLEN, NUMROWS)
dim alphaStat(NALPHA)
dim guess$ = ""
dim nguesses = 0
dim ngpl = 0
dim colors(2, 5)
dim istate = 0
dim ngreens = 0
dim running = 0

' Main Program
'open "debug.txt" for output as #1
ReadColors
Reset
ChooseWord
DrawIntroPage
HandleInput
end

' Read the colors for letter status (text, bg)
' 1: not yet guessed
' 2: guessed, word status still not known
' 3: guessed, not in word
' 4: guessed, in word but wrong position
' 5: guessed, in word and correct position
sub ReadColors
  local i, j
  for i = 1 to 5
    for j = 1 to 2
      read colors(j, i)
    next j
  next i
end sub

' Reset the puzzle
sub Reset
  local i, j
  for i = 1 to NUMROWS
    for j = 1 to WLEN  
      rowContent(j, i) = 1*1000
    next j
  next i
  for i = 1 to NALPHA
    alphaStat(i) = 1
  next i
  nguesses = 0
  guess$ = ""
  ngpl = 0
  ngreens = 0
  running = 1
end sub

' Choose a random 5-letter word from the dictionary
sub ChooseWord
  local wx, fp
  local buf$
  wx = RandomIntegerInRange(1, NDWRDS)
  fp = (wx-1)*TLEN + 1
  open dict$ for random as #2
  seek #2, fp
  line input #2, buf$
  close #2
  theWord$ = UCASE$(buf$)
end sub

' See if proffered word is a known word
' returns 1 if yes, 0 no
function IsAWord(g$)
  local wx, fp, ok
  local buf$, w$
  ok = 0
  open dict$ for random as #2
  for wx = 1 to NDWRDS
    fp = (wx-1)*TLEN + 1
    seek #2, fp
    line input #2, buf$
    w$ = UCASE$(LEFT$(buf$, WLEN))
    if w$ = g$ then
      ok = 1
      exit for
    end if
  next wx
  close #2
  IsAWord = ok
end function

' Draw the puzzle
sub DrawPuzzle
  DrawCells
  DrawAlphabet
end sub

' Draw the array of cells at the top of the puzzle
sub DrawCells
  local i, j, x, y, rc, a, st, c, tc
  y = PSY
  for i = 1 to NUMROWS
    x = PSX
    for j = 1 to WLEN
      rc = rowContent(j, i)
      st = rc\1000
      a = rc - 1000*st
      tc = colors(1, st)
      c = colors(2, st)
      box x, y, BWIDTH, BHEIGHT,, rgb(white), c
      if i <= nguesses then
        if a >= asc("A") then
          tc = colors(1, st)
          c = colors(2, st)
          text x+FBGAP+2, y+FBGAP+2, chr$(a), "LT", FNUM,, tc, -1
        end if
      end if
      inc x, BWIDTH + GAP
    next j
    inc y, BHEIGHT + GAP
  next i
end sub

' Draw the alphabet at the bottom of the puzzle
sub DrawAlphabet
  local x, y, row, col, ax, st, c, tc
  y = ALY
  ax = asc("A")
  for row = 1 to 2
    x = ALX
    for col = 1 to 13
      st = WasGuessed(chr$(ax))
      tc = colors(1, st)
      c = colors(2, st)
      box x, y, BWIDTH, BHEIGHT,, rgb(white), c
      text x+FBGAP, y+FBGAP+2, chr$(ax), "LT", FNUM,, tc, -1
      inc ax
      inc x, BWIDTH+GAP
    next col
    inc y, BHEIGHT+GAP
  next row
end sub    
            
' Handle user keystrokes
sub HandleInput
  local z$, g$
  local cmd, ucmd, i, ax
  do
    z$ = INKEY$
    do
      z$ = INKEY$
    loop until z$ <> ""
    ShowStatus ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case ENTER
        if running = 1 then
          if nguesses < numrows and  ngpl <> WLEN then
            ShowStatus "Word too short"
          else
            g$ = ""
            for i = 1 to WLEN
              st = rowContent(i, nguesses)\1000
              ax = rowContent(i, nguesses) - 1000*st
              g$ = g$ + chr$(ax)  
            next i
            k = IsAWord(g$)
            if k = 0 then
              ShowStatus "Not a Word"
            else
              UpdateGuessStatus
              if nguesses < numrows then inc nguesses
              ngpl = 0
              DrawPuzzle
            end if
          end if
        end if
      case ESC
        cls
        end
      case CAPA to CAPZ
        if running = 1 then
          if ngpl < WLEN then
            if ngpl = 0 and nguesses = 0 then inc nguesses
            inc ngpl
            guess$ = guess$ + chr$(cmd)
            rowContent(ngpl, nguesses) = 2000 + cmd
            DrawPuzzle
          end if
        end if
      case BACK
        if running = 1 then
          if ngpl > 0 then
            rowContent(ngpl, nguesses) = 1000
            inc ngpl, -1
            DrawPuzzle
          end if
        end if
      case HOME
        Reset
        ChooseWord
        cls
        DrawPuzzle
      case else
    end select
  loop    
end sub

' input is a letter and its position in the current guess row.
' returns 1 if the letter has not been guessed yet
' returns 2 if the letter has been guessed but word status still not known
' returns 3 if letter has been guessed but is not in word
' returns 4 if the letter has been guessed and was in word but out of place
' returns 5 if the letter has been guessed and was in the word in the correct place
function WasGuessed(a$)
  local ua = asc(UCASE$(a$))
  local ix = ua - asc("A") + 1
  WasGuessed = alphastat(ix)
end function

' Compare the list of guessed letters to theWord$ letters and mark both the alphastat array
' and the rowContent array values to reflect the letter status:
'   1: letter not yet guessed
'   2: letter guessed but word status not yet known
'   3: letter guessed but not in word
'   4: letter guessed and in word but out of place
'   5: letter guessed in in word and in place
' The default letter status is 3.
' If a letter exactly matches one in the word, the status is 5
' Otherwise, we count how many of that letter are in the word.
' If one or more, then we see which of the prior guess letters
' are the same and if so, if they were already have status 4.
' If we haven't already accounted for all the instances of the
' letter in the word, then we set status 4. (More complicated
' than first expected.)
sub UpdateGuessStatus
  local i, j, rc, a, s, n, ax, st, gp, ngreens, nyellows, nc
  local k, krc, ks, ka
  local cp(WLEN)
  local c$, kc$
  local m(WLEN)
  ngreens = 0
  for i = 1 to WLEN
    m(i) = 0
    st = 3
    rc = rowContent(i, nguesses)
    s = rc\1000
    a = rc - 1000*s
    c$ = chr$(a)
    ax = a - asc("A") + 1
    w$ = MID$(theWord$, i, 1)
    n = 0
    if c$ = w$ then
      st = 5
      m(i) = 1
      inc ngreens
    end if
    rowContent(i, nguesses) = st*1000 + a
    alphaStat(ax) = st
  next i
  for i = 1 to WLEN
    rc = rowContent(i, nguesses)
    s = rc\1000
    a = rc - 1000*s
    c$ = chr$(a)
    ax = a - asc("A") + 1
    nc = 0
    for j = 1 to WLEN
      if MID$(theWord$, j, 1) = c$ then inc nc
    next j
    if s < 4 then
      for j = 1 to WLEN
        w$ = MID$(theWord$, j, 1)
        nkc = 0
        for k = 1 to j-1
          krc = rowContent(k, nguesses)
          ks = krc\1000
          ka = krc - 1000*ks
          kc$ = chr$(ka)
          if kc$ = c$ and ks = 4 then inc nkc
        next k
        if c$ = w$ and m(j) = 0 and nkc < nc then
          st = 4
          rowContent(i, nguesses) = st*1000 + a
          alphaStat(ax) = st
        end if
      next j
    end if
  next i
  CheckEndGame ngreens
end sub

' See if game is over
sub CheckEndGame ngreens
  if ngreens = WLEN then
    running = 0
    ShowStatus "You Win!"
  else if nguesses = numrows then
    running = 0
    if ngreens = WLEN then
      ShowStatus "You Win!"
    else
      ShowStatus "Sorry, you lose. The word was '" + theWord$ + "'"
    end if
  end if
  if running = 0 then
    text MM.HRES\2, 570, "Press Home for new game", "CT"    
  end if
end sub

' Introduction Page
sub DrawIntroPage
  local x, y, w
  local z$
  cls
  w = 400
  x = MM.HRES\2
  text x, 20, "WORDLE!", "CT", 5,, rgb(green)
  x = x - w\2
  y = 60 : w = 400
  box x, y, w, 180,, rgb(white), rgb(white)
  inc x, 20
  inc y, 10
  text x, y, "Guess the Word!", "LT", 1,, rgb(black), -1
  inc y, 15
  text x, y, "Write a 5-letter word and press Enter.", "LT", 1,, rgb(black), -1
  inc y, 30
  text x, y, "Green = correct spot.", "LT", 1,, rgb(black), rgb(green)
  inc y, 15
  text x, y, "Yellow = wrong spot.", "LT", 1,, rgb(black), rgb(yellow) 
  inc y, 15
  text x, y, "Black = not in word.", "LT", 1,, rgb(white), rgb(black)
  inc y, 30
  text x, y, "Press Home for a new game.", "LT", 1,, rgb(black), -1
  inc y, 15
  text x, y, "Press Escape to quit.", "LT", 1,, rgb(black), -1
  inc y, 30
  text x, y, "Press any key to play.", "LT", 1,, rgb(black), -1
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cls
  DrawPuzzle
end sub

' print a status message at the bottom of the puzzle.
' The message persists until the next keystroke.
sub ShowStatus msg$
  text MM.HRES\2, 550, space$(60), "CT"
  text MM.HRES\2, 550, msg$, "CT"
end sub

' Generate a uniformly distributed random integer in the closed range a to b
' (gets around issues with non-uniform distribution in rnd() at some
' expense in performance.)
function RandomIntegerInRange(a as integer, b as integer) as integer
  local v, c
  c = b - a + 1
  do
    v = a + (b-a+2)*rnd()
    if v > 1 and v-a <= c then exit do
  loop
  RandomIntegerInRange = v-1
end function

' Colors for various letter statuses (text, background)
data rgb(black), rgb(gray)
data rgb(white), rgb(gray)
data rgb(white), rgb(black)
data rgb(black), rgb(255,255,0)
data rgb(black), rgb(green)

